//////////////////////////////////////////////////////////////////////////////
//
//  Copyright 2013 by Autodesk, Inc.  All rights reserved.
//
// This computer source code and related instructions and comments
// are the unpublished confidential and proprietary information of
// Autodesk, Inc. and are protected under applicable copyright and
// trade secret law.  They may not be disclosed to, copied or used
// by any third party without the prior written consent of Autodesk, Inc.
//
//////////////////////////////////////////////////////////////////////////////

$.Controller('CoordSysController',    // coord_sys
    {
        InputState: { ReadOnlyState: 0, EdittingState: 1 },
        location: { longitude: null, latitude: null, elevation: 0.0 },
        existingCurrentCSId: null,
        currentCSId: null,
        $originalTableRows: null,
        tableRowsSortFunc: null,
        timeZoneSuggest: false,
        //static methods
        executeNativeSync: function (param) {
            return exec(JSON.stringify(param));
        },
        executeNative: function executeNative(param, callback) {
            execAsync(JSON.stringify(param),
            function (result) {
                if (callback) callback(result);
            },
            function (result) {
                setLoadingState(1);
                alert($.localize("ID_CS_LIBRARY_NOT_AVAILABLE", SELECTED_LANG));
            });
        },            
        // Meant for handling keydown input events. Returns true if event was handled
        handleKeyDownEvent: function(event) {
            if (event.keyCode == 38) { // ARROW UP
                if(document.activeElement == document.body || document.activeElement == $('#csSearch').get(0)) {
                    CoordSysController.handleCsTableArrowUp();
                    event.preventDefault();
                }
                return true;
            } else if (event.keyCode == 40) { // ARROW DOWN
                if(document.activeElement == document.body || document.activeElement == $('#csSearch').get(0)) {
                    CoordSysController.handleCsTableArrowDown();
                    event.preventDefault();
                }
                return true;
            }

            return false;
        },
        handleCsTableArrowUp: function() {
            var $selectedRow = $('#csTableDiv tbody tr.tableRowSelected');
            var $prevRow = $selectedRow.prev();
            if($prevRow.length > 0) {
                // deselect current row and select previous row
                $selectedRow.removeClass('tableRowSelected');
                $prevRow.addClass('tableRowSelected');
                
                // display newly selected value
                var csNameSelected = $prevRow.find('td:first').text();                
                $('#coordSysSection').coord_sys('setCurrentCoordinateSystem', csNameSelected);
            
                // if previous row is out of table display, scroll it into view!
                var $csTableDiv = $('#csTableDiv');
                var scrollTop = $csTableDiv.scrollTop()

                var relativeRowOffset = $prevRow.offset().top - $csTableDiv.offset().top;
                var rowHeight = $prevRow.height();

                if(relativeRowOffset < 0) {
                    $csTableDiv.scrollTop(scrollTop - rowHeight * 2.0);
                }
            }
        },
        handleCsTableArrowDown: function() {
            var $selectedRow = $('#csTableDiv tbody tr.tableRowSelected');
            var $nextRow = $selectedRow.next();
            if($nextRow.length > 0) {
                // deselect current row and select next row
                $selectedRow.removeClass('tableRowSelected');
                $nextRow.addClass('tableRowSelected');
                
                // display newly selected value
                var csNameSelected = $nextRow.find('td:first').text();
                $('#coordSysSection').coord_sys('setCurrentCoordinateSystem', csNameSelected);  

                // if next row is out of table display, scroll it into view!
                var $csTableDiv = $('#csTableDiv');
                var scrollTop = $csTableDiv.scrollTop()

                var relativeRowOffset = $nextRow.offset().top - $csTableDiv.offset().top;
                var rowHeight = $nextRow.height();

                if(relativeRowOffset + rowHeight > $csTableDiv.height()) {
                    $csTableDiv.scrollTop(scrollTop + rowHeight * 2.0);
                }
            }
        },
        getCoordinateSystem: function (lat, long, callback) {
            var args = { 'functionName': 'Ac_Geo.filterCoordinateSystem', 'invokeAsCommand': false, 'functionParams': { 'longitude': long, 'latitude': lat} };
            CoordSysController.executeNative(args, callback);
        },
        updateLocationandCoordinateSystem: function () {
            // inform native side of the coordinate system change
            var latitude = CoordSysController.location.latitude == null ? 0 : CoordSysController.location.latitude;
            var longitude = CoordSysController.location.longitude == null ? 0 : CoordSysController.location.longitude;
            var elevation = parseFloat($('#elevation').val());
            if (isNaN(elevation))
                elevation = 0.0;

            var args = { 'functionName': 'Ac_Geo.updateLocationAndCoordinateSystem',
                'invokeAsCommand': false,
                'functionParams': {
                    'latitude': latitude,
                    'longitude': longitude,
                    'elevation': elevation,
                    'coordinateSystemId': CoordSysController.currentCSId
                }
            };
            CoordSysController.executeNative(args);
        },
        populateCoordinateSystem: function (result) {
            var res = JSON.parse(result);

            var tableRows = "";

            if (res && res.coords_system) {
                for (var i = 0; i < res.coords_system.length; ++i) {
                    var coord = res.coords_system[i];
                    if (coord.id && coord.unit && coord.referenceto) {
                        var id = coord.id;
                        var epsg = coord.epsg;

                        // show epsg as dash if it's value is zero'
                        if(epsg == 0)
                            epsg = '-';

                        var selectedClass = "";

                        // if drawing already has a coordinate system, we show it as selected
                        if (CoordSysController.existingCurrentCSId != null && CoordSysController.existingCurrentCSId == coord.id)
                        {
                            selectedClass = 'tableRowSelected';
                        }

                        // append row to table
                        var eachTableRow = '<tr class="' + selectedClass + '"><td class="csColumnName">' + id + '</td><td class="csColumnReference">' + coord.referenceto +
                                           '</td><td class="csColumnUnit">' + coord.unit + '</td><td class="csColumnEPSG">' + epsg + '</td></tr>';
                        tableRows += eachTableRow;
                    }
                }
            }

            $('#csTableDiv tbody').html(tableRows);

            CoordSysController.$originalTableRows = $(tableRows);

            CoordSysController.tableRowsSortFunc = null; // default no sorting

            if (CoordSysController.existingCurrentCSId == null) {
                CoordSysController.disableSaveButton();
            }
            else {
                CoordSysController.updateLocationandCoordinateSystem();
                CoordSysController.enableTimeZoneSel();
                CoordSysController.enableUnitsSel();
                CoordSysController.enableNextButton();
            }

            // compute the time zone if location was explicitly changed
            var markerLocController = $('#markerInfoAndListCS').controllers(MarkerLocationController)[0];
            if(markerLocController.locationChanged) {
                CoordSysController.suggestTimeZone(CoordSysController.location.longitude);
            }

            setLoadingState(1);

            $('#csSearch').focus();
        },
        populateTimezones: function (result) {
            result = result.replace("'", "\'");
            var res = JSON.parse(result);
            if (res && res.alltimezones) {
                $('#timeZone').empty();
                for (var i = 0; i < res.alltimezones.length; ++i) {
                    var opt = $('<option>' + res.alltimezones[i].desc + '</option>');
                    opt.attr('value', Math.floor(res.alltimezones[i].timezone));
                    opt.attr('offset', Math.floor(res.alltimezones[i].offset));
                    $('#timeZone').append(opt);
                }

                if (globalQueries.hasOwnProperty('timezone')) {
                    var timeZone = globalQueries['timezone'].toString();
                    $('#timeZone').val(timeZone);
                }
            }
        },
        getTimeZones: function () {
            var args = { 'functionName': 'Ac_Geo.getTimeZones', 'invokeAsCommand': false, 'functionParams': {} };
            CoordSysController.executeNative(args, CoordSysController.populateTimezones);
        },

        populateUnits: function (result) {
            result = result.replace("'", "\'");
            var res = JSON.parse(result);
            if (res && res.units) {
                $('#listDrawingUnits').empty();
                for (var i = 0; i < res.units.length; ++i) {
                    var opt = $('<option>' + res.units[i].unitdesc + '</option>');
                    opt.attr('value', res.units[i].unitvalue.toString());
                    $('#listDrawingUnits').append(opt);
                }
                if (res.defaultunit)
                    $('#listDrawingUnits').val(res.defaultunit.toString());

                $('#listDrawingUnits').change(function () {
                    // Call native to check whether unit is mismatch with INSUNIT and prompt alert if yes
                    var unit = $('#listDrawingUnits').val();
                    var args = { 'functionName': 'Ac_Geo.listDrawingUnitsOnChanged', 'invokeAsCommand': false, 'functionParams': {'unit': unit } };
                    var jsonStr = exec(JSON.stringify(args));
                    jsonStr = jsonStr.replace("'", "\'");
                    var res = JSON.parse(jsonStr);
                    // reset the unit
                    if (res.unit > 0)
                        $('#listDrawingUnits').val(res.unit);
                });
            }
        },
        getUnits: function () {
            var args = { 'functionName': 'Ac_Geo.getUnits', 'invokeAsCommand': false, 'functionParams': {} };
            CoordSysController.executeNative(args, CoordSysController.populateUnits);
        },
        enableNextButton: function () {
            var s = $('input.invalid');
            var isEdittingState = $('#btneditok').is(":visible");
            if (s.length > 0 || isEdittingState)
                $("#buttonsRow").data("controllers")['buttons_row'].setNextButtonEnabled(false);
            else
                $("#buttonsRow").data("controllers")['buttons_row'].setNextButtonEnabled(true);
        },
        disableSaveButton: function () {
            $("#buttonsRow").data("controllers")['buttons_row'].setNextButtonEnabled(false);
        },
        enableTimeZoneSel: function () {
            $('#timeZone').attr("disabled", false);
        },
        enableUnitsSel: function () {
            $('#listDrawingUnits').attr("disabled", false);
        },
        suggestTimeZone: function (longitude) {
            var offset = Math.floor(CoordSysController.getTimeZone(longitude));
            $('#timeZone').find('option[offset="' + offset + '"]:first').attr("selected", true);
            CoordSysController.enableTimeZoneSel();
        },
        getTimeZone: function (longitude) {
            var tz;
            if (longitude <= 0) {
                tz = -Math.floor((-longitude + 7.5) / 15);
            }
            else {
                tz = Math.floor((longitude + 7.5) / 15);
            }
            return tz;
        },
        sortFilterTable: function () {
            var $sortedTableRows = CoordSysController.$originalTableRows;
                                
            // 1st - we sort
            // we use original rows to keep the sort from changing when keeping clicking on the header
            if(CoordSysController.tableRowsSortFunc)
                $sortedTableRows = CoordSysController.$originalTableRows.sort(CoordSysController.tableRowsSortFunc);
                
            // because originalTableRows has the initial coordinate system selected, we need to clear it
            // and select the row corresponding to the current coordinate system
            $sortedTableRows.removeClass('tableRowSelected');
            $sortedTableRows.filter(function(index) {
                                          return $(this).find('td:first').text() == CoordSysController.currentCSId;
                                      }).addClass('tableRowSelected');                  

            // 2nd - we filter sorted list by search string
            var searchStr = $('#csSearch').val().toLowerCase();
            var filterFunc = function(index) {
                                return true; // no filtering
                             };

            if (searchStr) {
                // filter the coordinate systems rows
                filterFunc = function(index) {
                    // 'this' instanceof tr
                    var name = $('td.csColumnName', this).text().toLowerCase();
                    var ref = $('td.csColumnReference', this).text().toLowerCase();
                    var unit = $('td.csColumnUnit', this).text().toLowerCase();
                    var epsg = $('td.csColumnEPSG', this).text().toLowerCase();

                    if(searchStr.length == 1 && /^[a-z]+$/i.test(searchStr)) {
                        if(name.indexOf(searchStr) == 0)
                            return true;
                    }
                    else {
                        if(name.indexOf(searchStr) >= 0)
                            return true;
                        else if(ref.indexOf(searchStr) >= 0)
                            return true;
                        else if(unit.indexOf(searchStr) >= 0)
                            return true;
                        else if(epsg.indexOf(searchStr) >= 0)
                            return true;
                    }

                    return false;
                };
            }


            var $filteredTableRows = $sortedTableRows.filter(filterFunc);                      

            $('#csTableDiv tbody').html($filteredTableRows);
        }
    },
//instance methods
    {
    init: function (element, options) {
        $('#coordSysSection').html('coord_sys_template', {});

        $('#coordinateSystemLabel').html($.localize("ID_COORDINATE_SYSTEM", SELECTED_LANG));
        $('#timeZoneLabel').html($.localize("ID_TIMEZONE", SELECTED_LANG));
        $('#drawingUnitLabel').html($.localize("ID_DRAWING_UNIT", SELECTED_LANG));

        CoordSysController.existingCurrentCSId = null;
        if (globalQueries.hasOwnProperty('curCSId')) {
            CoordSysController.existingCurrentCSId = globalQueries['curCSId'];

            $('#coordinateSystemText').html(CoordSysController.existingCurrentCSId);

            CoordSysController.currentCSId = CoordSysController.existingCurrentCSId;
        }

        //retrieve timezone and units from autocad
        CoordSysController.getTimeZones();
        CoordSysController.getUnits();

        // Enable "Continue..." button if data is populated from KMLKMZ file
        if (globalQueries.hasOwnProperty('fromKML')) {
            if (globalQueries['fromKML'] > 0)
                CoordSysController.enableNextButton();
        }

        // this enables the watermark to disappear when type into search box
        $('label').marktext();

        // populate coordinate systems table
        this.setLocation(MarkerLocationController.location);
    },
    destroy: function () {
        this._super();
    },
    setCurrentCoordinateSystem: function (csName) {
        this.find('#coordinateSystemText').html(csName);

        CoordSysController.currentCSId = csName;

        CoordSysController.updateLocationandCoordinateSystem();
        CoordSysController.enableTimeZoneSel();
        CoordSysController.enableUnitsSel();
        CoordSysController.enableNextButton();            
    },        
    setLocation: function (location) {
        setLoadingState(0);
        var self = this;
        setTimeout(function () {
            CoordSysController.location.longitude = location.longitude;
            CoordSysController.location.latitude = location.latitude;
            CoordSysController.location.elevation = location.elevation;

            CoordSysController.getCoordinateSystem(location.latitude, location.longitude, CoordSysController.populateCoordinateSystem);

        }, 200);
    },
    '#csSearch keyup': function (el, ev) {
        if ( ev.keyCode === 9  ||  //shift
             ev.keyCode === 16 ||  //tab
             ev.keyCode === 37 ||  //arrow left
             ev.keyCode === 38 ||  //arrow up
             ev.keyCode === 39 ||  //arrow right
             ev.keyCode === 40)    //arrow down
             return;
             
        CoordSysController.sortFilterTable();
    },
    '#csSearch cut': function (el, ev) {        
        setTimeout(function () {
            CoordSysController.sortFilterTable();        
        }, 100);                
    },      
    '#csSearch paste': function (el, ev) {        
        setTimeout(function () {
            CoordSysController.sortFilterTable();        
        }, 100);                
    },    
    // selection of coordinate system through table row mouse click
    '#csTableDiv tr click': function (el, ev) {
        $('#csTableDiv tr').removeClass('tableRowSelected');
        el.addClass('tableRowSelected');

        var csNameSelected = el.find('td:first').text();

        this.setCurrentCoordinateSystem(csNameSelected);
    },
    '#csTableHeaderDiv thead th click': function (el, ev) {
        var columnIndex = el.index();

        $spans = $('#csTableHeaderDiv thead th span');
        $spans.removeClass('csHeaderSortUp');
        $spans.removeClass('csHeaderSortDown');

        if(el.hasClass('csHeaderSortedGreater')) {
            el.removeClass('csHeaderSortedGreater');
            el.children('span').addClass('csHeaderSortDown');

            CoordSysController.tableRowsSortFunc = function (a,b) {
                                                        var aCsName = $(a).find('td').eq(columnIndex).text();
                                                        var bCsName = $(b).find('td').eq(columnIndex).text();

                                                        return aCsName.toLowerCase() < bCsName.toLowerCase() ? 1 : -1;
                                                    };
        }
        else {
            el.addClass('csHeaderSortedGreater');
            el.children('span').addClass('csHeaderSortUp');

            CoordSysController.tableRowsSortFunc = function (a,b) {
                                                        var aCsName = $(a).find('td').eq(columnIndex).text();
                                                        var bCsName = $(b).find('td').eq(columnIndex).text();

                                                        return aCsName.toLowerCase() > bCsName.toLowerCase() ? 1 : -1;
                                                     };
        }

        CoordSysController.sortFilterTable();
    }
});



// SIG // Begin signature block
// SIG // MIIZNgYJKoZIhvcNAQcCoIIZJzCCGSMCAQExCzAJBgUr
// SIG // DgMCGgUAMGcGCisGAQQBgjcCAQSgWTBXMDIGCisGAQQB
// SIG // gjcCAR4wJAIBAQQQEODJBs441BGiowAQS9NQkAIBAAIB
// SIG // AAIBAAIBAAIBADAhMAkGBSsOAwIaBQAEFNsJDYhDL7zn
// SIG // DFpBqlCmZBmxV35XoIIUMDCCA+4wggNXoAMCAQICEH6T
// SIG // 6/t8xk5Z6kuad9QG/DswDQYJKoZIhvcNAQEFBQAwgYsx
// SIG // CzAJBgNVBAYTAlpBMRUwEwYDVQQIEwxXZXN0ZXJuIENh
// SIG // cGUxFDASBgNVBAcTC0R1cmJhbnZpbGxlMQ8wDQYDVQQK
// SIG // EwZUaGF3dGUxHTAbBgNVBAsTFFRoYXd0ZSBDZXJ0aWZp
// SIG // Y2F0aW9uMR8wHQYDVQQDExZUaGF3dGUgVGltZXN0YW1w
// SIG // aW5nIENBMB4XDTEyMTIyMTAwMDAwMFoXDTIwMTIzMDIz
// SIG // NTk1OVowXjELMAkGA1UEBhMCVVMxHTAbBgNVBAoTFFN5
// SIG // bWFudGVjIENvcnBvcmF0aW9uMTAwLgYDVQQDEydTeW1h
// SIG // bnRlYyBUaW1lIFN0YW1waW5nIFNlcnZpY2VzIENBIC0g
// SIG // RzIwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
// SIG // AQCxrLNJVEuXHBIK2CV5kSJXKm/cuCbEQ3Nrwr8uUFr7
// SIG // FMJ2jkMBJUO0oeJF9Oi3e8N0zCLXtJQAAvdN7b+0t0Qk
// SIG // a81fRTvRRM5DEnMXgotptCvLmR6schsmTXEfsTHd+1Fh
// SIG // AlOmqvVJLAV4RaUvic7nmef+jOJXPz3GktxK+Hsz5HkK
// SIG // +/B1iEGc/8UDUZmq12yfk2mHZSmDhcJgFMTIyTsU2sCB
// SIG // 8B8NdN6SIqvK9/t0fCfm90obf6fDni2uiuqm5qonFn1h
// SIG // 95hxEbziUKFL5V365Q6nLJ+qZSDT2JboyHylTkhE/xni
// SIG // RAeSC9dohIBdanhkRc1gRn5UwRN8xXnxycFxAgMBAAGj
// SIG // gfowgfcwHQYDVR0OBBYEFF+a9W5czMx0mtTdfe8/2+xM
// SIG // gC7dMDIGCCsGAQUFBwEBBCYwJDAiBggrBgEFBQcwAYYW
// SIG // aHR0cDovL29jc3AudGhhd3RlLmNvbTASBgNVHRMBAf8E
// SIG // CDAGAQH/AgEAMD8GA1UdHwQ4MDYwNKAyoDCGLmh0dHA6
// SIG // Ly9jcmwudGhhd3RlLmNvbS9UaGF3dGVUaW1lc3RhbXBp
// SIG // bmdDQS5jcmwwEwYDVR0lBAwwCgYIKwYBBQUHAwgwDgYD
// SIG // VR0PAQH/BAQDAgEGMCgGA1UdEQQhMB+kHTAbMRkwFwYD
// SIG // VQQDExBUaW1lU3RhbXAtMjA0OC0xMA0GCSqGSIb3DQEB
// SIG // BQUAA4GBAAMJm495739ZMKrvaLX64wkdu0+CBl03X6ZS
// SIG // nxaN6hySCURu9W3rWHww6PlpjSNzCxJvR6muORH4KrGb
// SIG // sBrDjutZlgCtzgxNstAxpghcKnr84nodV0yoZRjpeUBi
// SIG // JZZux8c3aoMhCI5B6t3ZVz8dd0mHKhYGXqY4aiISo1EZ
// SIG // g362MIIEozCCA4ugAwIBAgIQDs/0OMj+vzVuBNhqmBsa
// SIG // UDANBgkqhkiG9w0BAQUFADBeMQswCQYDVQQGEwJVUzEd
// SIG // MBsGA1UEChMUU3ltYW50ZWMgQ29ycG9yYXRpb24xMDAu
// SIG // BgNVBAMTJ1N5bWFudGVjIFRpbWUgU3RhbXBpbmcgU2Vy
// SIG // dmljZXMgQ0EgLSBHMjAeFw0xMjEwMTgwMDAwMDBaFw0y
// SIG // MDEyMjkyMzU5NTlaMGIxCzAJBgNVBAYTAlVTMR0wGwYD
// SIG // VQQKExRTeW1hbnRlYyBDb3Jwb3JhdGlvbjE0MDIGA1UE
// SIG // AxMrU3ltYW50ZWMgVGltZSBTdGFtcGluZyBTZXJ2aWNl
// SIG // cyBTaWduZXIgLSBHNDCCASIwDQYJKoZIhvcNAQEBBQAD
// SIG // ggEPADCCAQoCggEBAKJjCzlEuLsjp0RJuw7/ofBhClOT
// SIG // sJjbrSwPSsVu/4Y8U1UPFc4EPyv9qZaW2b5heQtbyUyG
// SIG // duXgQ0sile7CK0PBn9hotI5AT+6FOLkRxSPyZFjwFTJv
// SIG // TlehroikAtcqHs1L4d1j1ReJMluwXplaqJ0oUA4X7pbb
// SIG // YTtFUR3PElYLkkf8q672Zj1HrHBy55LnX80QucSDZJQZ
// SIG // vSWA4ejSIqXQugJ6oXeTW2XD7hd0vEGGKtwITIySjJEt
// SIG // nndEH2jWqHR32w5bMotWizO92WPISZ06xcXqMwvS8aMb
// SIG // 9Iu+2bNXizveBKd6IrIkri7HcMW+ToMmCPsLvalPmQjh
// SIG // EChyqs0CAwEAAaOCAVcwggFTMAwGA1UdEwEB/wQCMAAw
// SIG // FgYDVR0lAQH/BAwwCgYIKwYBBQUHAwgwDgYDVR0PAQH/
// SIG // BAQDAgeAMHMGCCsGAQUFBwEBBGcwZTAqBggrBgEFBQcw
// SIG // AYYeaHR0cDovL3RzLW9jc3Aud3Muc3ltYW50ZWMuY29t
// SIG // MDcGCCsGAQUFBzAChitodHRwOi8vdHMtYWlhLndzLnN5
// SIG // bWFudGVjLmNvbS90c3MtY2EtZzIuY2VyMDwGA1UdHwQ1
// SIG // MDMwMaAvoC2GK2h0dHA6Ly90cy1jcmwud3Muc3ltYW50
// SIG // ZWMuY29tL3Rzcy1jYS1nMi5jcmwwKAYDVR0RBCEwH6Qd
// SIG // MBsxGTAXBgNVBAMTEFRpbWVTdGFtcC0yMDQ4LTIwHQYD
// SIG // VR0OBBYEFEbGaaMOShQe1UzaUmMXP142vA3mMB8GA1Ud
// SIG // IwQYMBaAFF+a9W5czMx0mtTdfe8/2+xMgC7dMA0GCSqG
// SIG // SIb3DQEBBQUAA4IBAQB4O7SRKgBM8I9iMDd4o4QnB28Y
// SIG // st4l3KDUlAOqhk4ln5pAAxzdzuN5yyFoBtq2MrRtv/Qs
// SIG // JmMz5ElkbQ3mw2cO9wWkNWx8iRbG6bLfsundIMZxD82V
// SIG // dNy2XN69Nx9DeOZ4tc0oBCCjqvFLxIgpkQ6A0RH83Vx2
// SIG // bk9eDkVGQW4NsOo4mrE62glxEPwcebSAe6xp9P2ctgwW
// SIG // K/F/Wwk9m1viFsoTgW0ALjgNqCmPLOGy9FqpAa8VnCwv
// SIG // SRvbIrvD/niUUcOGsYKIXfA9tFGheTMrLnu53CAJE3Hr
// SIG // ahlbz+ilMFcsiUk/uc9/yb8+ImhjU5q9aXSsxR08f5Lg
// SIG // w7wc2AR1MIIFhTCCBG2gAwIBAgIQKcFbP6rNUmpOZ708
// SIG // Tn4/8jANBgkqhkiG9w0BAQUFADCBtDELMAkGA1UEBhMC
// SIG // VVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYD
// SIG // VQQLExZWZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTswOQYD
// SIG // VQQLEzJUZXJtcyBvZiB1c2UgYXQgaHR0cHM6Ly93d3cu
// SIG // dmVyaXNpZ24uY29tL3JwYSAoYykxMDEuMCwGA1UEAxMl
// SIG // VmVyaVNpZ24gQ2xhc3MgMyBDb2RlIFNpZ25pbmcgMjAx
// SIG // MCBDQTAeFw0xMjA3MjUwMDAwMDBaFw0xNTA5MjAyMzU5
// SIG // NTlaMIHIMQswCQYDVQQGEwJVUzETMBEGA1UECBMKQ2Fs
// SIG // aWZvcm5pYTETMBEGA1UEBxMKU2FuIFJhZmFlbDEWMBQG
// SIG // A1UEChQNQXV0b2Rlc2ssIEluYzE+MDwGA1UECxM1RGln
// SIG // aXRhbCBJRCBDbGFzcyAzIC0gTWljcm9zb2Z0IFNvZnR3
// SIG // YXJlIFZhbGlkYXRpb24gdjIxHzAdBgNVBAsUFkRlc2ln
// SIG // biBTb2x1dGlvbnMgR3JvdXAxFjAUBgNVBAMUDUF1dG9k
// SIG // ZXNrLCBJbmMwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAw
// SIG // ggEKAoIBAQCoYmDrmd0Gq8ezSsDlfgaJFEFplNPNhWzM
// SIG // 2uFQaYAB/ggpQ11+N4B6ao+TqrNIWDIqt3JKhaU889nx
// SIG // l/7teWGwuOurstI2Z0bEDhXiXam/bicK2HVLyntliQ+6
// SIG // tT+nlgfN8tgB2NzM0BpE1YCnU2b6DwQw4V7BV+/F//83
// SIG // yGFOpePlumzXxNw9EKWkaq81slmmTxf7UxZgP9PGbLw8
// SIG // gLAPk4PTJI97+5BBqhkLb1YqSfWn3PNMfsNKhw/VwAN0
// SIG // dRKeM6H8SkOdz+osr+NyH86lsKQuics4fwK5uFSHQHsI
// SIG // t6Z0tqWvminRqceUi9ugRlGryh9X1ZqCqfL/ggdzYa3Z
// SIG // AgMBAAGjggF7MIIBdzAJBgNVHRMEAjAAMA4GA1UdDwEB
// SIG // /wQEAwIHgDBABgNVHR8EOTA3MDWgM6Axhi9odHRwOi8v
// SIG // Y3NjMy0yMDEwLWNybC52ZXJpc2lnbi5jb20vQ1NDMy0y
// SIG // MDEwLmNybDBEBgNVHSAEPTA7MDkGC2CGSAGG+EUBBxcD
// SIG // MCowKAYIKwYBBQUHAgEWHGh0dHBzOi8vd3d3LnZlcmlz
// SIG // aWduLmNvbS9ycGEwEwYDVR0lBAwwCgYIKwYBBQUHAwMw
// SIG // cQYIKwYBBQUHAQEEZTBjMCQGCCsGAQUFBzABhhhodHRw
// SIG // Oi8vb2NzcC52ZXJpc2lnbi5jb20wOwYIKwYBBQUHMAKG
// SIG // L2h0dHA6Ly9jc2MzLTIwMTAtYWlhLnZlcmlzaWduLmNv
// SIG // bS9DU0MzLTIwMTAuY2VyMB8GA1UdIwQYMBaAFM+Zqep7
// SIG // JvRLyY6P1/AFJu/j0qedMBEGCWCGSAGG+EIBAQQEAwIE
// SIG // EDAWBgorBgEEAYI3AgEbBAgwBgEBAAEB/zANBgkqhkiG
// SIG // 9w0BAQUFAAOCAQEA2OkGvuiY7TyI6yVTQAYmTO+MpOFG
// SIG // C8MflHSbofJiuLxrS1KXbkzsAPFPPsU1ouftFhsXFtDQ
// SIG // 8rMTq/jwugTpbJUREV0buEkLl8AKRhYQTKBKg1I/puBv
// SIG // bkJocDE0pRwtBz3xSlXXEwyYPcbCOnrM3OZ5bKx1Qiii
// SIG // vixlcGWhO3ws904ssutPFf4mV5PDi3U2Yp1HgbBK/Um/
// SIG // FLr6YAYeZaA8KY1CfQEisF3UKTwm72d7S+fJf++SOGea
// SIG // K0kumehVcbavQJTOVebuZ9V+qU0nk1lMrqve9BnQK69B
// SIG // QqNZu77vCO0wm81cfynAxkOYKZG3idY47qPJOgXKkwmI
// SIG // 2+92ozCCBgowggTyoAMCAQICEFIA5aolVvwahu2WydRL
// SIG // M8cwDQYJKoZIhvcNAQEFBQAwgcoxCzAJBgNVBAYTAlVT
// SIG // MRcwFQYDVQQKEw5WZXJpU2lnbiwgSW5jLjEfMB0GA1UE
// SIG // CxMWVmVyaVNpZ24gVHJ1c3QgTmV0d29yazE6MDgGA1UE
// SIG // CxMxKGMpIDIwMDYgVmVyaVNpZ24sIEluYy4gLSBGb3Ig
// SIG // YXV0aG9yaXplZCB1c2Ugb25seTFFMEMGA1UEAxM8VmVy
// SIG // aVNpZ24gQ2xhc3MgMyBQdWJsaWMgUHJpbWFyeSBDZXJ0
// SIG // aWZpY2F0aW9uIEF1dGhvcml0eSAtIEc1MB4XDTEwMDIw
// SIG // ODAwMDAwMFoXDTIwMDIwNzIzNTk1OVowgbQxCzAJBgNV
// SIG // BAYTAlVTMRcwFQYDVQQKEw5WZXJpU2lnbiwgSW5jLjEf
// SIG // MB0GA1UECxMWVmVyaVNpZ24gVHJ1c3QgTmV0d29yazE7
// SIG // MDkGA1UECxMyVGVybXMgb2YgdXNlIGF0IGh0dHBzOi8v
// SIG // d3d3LnZlcmlzaWduLmNvbS9ycGEgKGMpMTAxLjAsBgNV
// SIG // BAMTJVZlcmlTaWduIENsYXNzIDMgQ29kZSBTaWduaW5n
// SIG // IDIwMTAgQ0EwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAw
// SIG // ggEKAoIBAQD1I0tepdeKuzLp1Ff37+THJn6tGZj+qJ19
// SIG // lPY2axDXdYEwfwRof8srdR7NHQiM32mUpzejnHuA4Jnh
// SIG // 7jdNX847FO6G1ND1JzW8JQs4p4xjnRejCKWrsPvNamKC
// SIG // TNUh2hvZ8eOEO4oqT4VbkAFPyad2EH8nA3y+rn59wd35
// SIG // BbwbSJxp58CkPDxBAD7fluXF5JRx1lUBxwAmSkA8taEm
// SIG // qQynbYCOkCV7z78/HOsvlvrlh3fGtVayejtUMFMb32I0
// SIG // /x7R9FqTKIXlTBdOflv9pJOZf9/N76R17+8V9kfn+Bly
// SIG // 2C40Gqa0p0x+vbtPDD1X8TDWpjaO1oB21xkupc1+NC2J
// SIG // AgMBAAGjggH+MIIB+jASBgNVHRMBAf8ECDAGAQH/AgEA
// SIG // MHAGA1UdIARpMGcwZQYLYIZIAYb4RQEHFwMwVjAoBggr
// SIG // BgEFBQcCARYcaHR0cHM6Ly93d3cudmVyaXNpZ24uY29t
// SIG // L2NwczAqBggrBgEFBQcCAjAeGhxodHRwczovL3d3dy52
// SIG // ZXJpc2lnbi5jb20vcnBhMA4GA1UdDwEB/wQEAwIBBjBt
// SIG // BggrBgEFBQcBDARhMF+hXaBbMFkwVzBVFglpbWFnZS9n
// SIG // aWYwITAfMAcGBSsOAwIaBBSP5dMahqyNjmvDz4Bq1EgY
// SIG // LHsZLjAlFiNodHRwOi8vbG9nby52ZXJpc2lnbi5jb20v
// SIG // dnNsb2dvLmdpZjA0BgNVHR8ELTArMCmgJ6AlhiNodHRw
// SIG // Oi8vY3JsLnZlcmlzaWduLmNvbS9wY2EzLWc1LmNybDA0
// SIG // BggrBgEFBQcBAQQoMCYwJAYIKwYBBQUHMAGGGGh0dHA6
// SIG // Ly9vY3NwLnZlcmlzaWduLmNvbTAdBgNVHSUEFjAUBggr
// SIG // BgEFBQcDAgYIKwYBBQUHAwMwKAYDVR0RBCEwH6QdMBsx
// SIG // GTAXBgNVBAMTEFZlcmlTaWduTVBLSS0yLTgwHQYDVR0O
// SIG // BBYEFM+Zqep7JvRLyY6P1/AFJu/j0qedMB8GA1UdIwQY
// SIG // MBaAFH/TZafC3ey78DAJ80M5+gKvMzEzMA0GCSqGSIb3
// SIG // DQEBBQUAA4IBAQBWIuY0pMRhy0i5Aa1WqGQP2YyRxLvM
// SIG // DOWteqAif99HOEotbNF/cRp87HCpsfBP5A8MU/oVXv50
// SIG // mEkkhYEmHJEUR7BMY4y7oTTUxkXoDYUmcwPQqYxkbdxx
// SIG // kuZFBWAVWVE5/FgUa/7UpO15awgMQXLnNyIGCb4j6T9E
// SIG // mh7pYZ3MsZBc/D3SjaxCPWU21LQ9QCiPmxDPIybMSyDL
// SIG // kB9djEw0yjzY5TfWb6UgvTTrJtmuDefFmvehtCGRM2+G
// SIG // 6Fi7JXx0Dlj+dRtjP84xfJuPG5aexVN2hFucrZH6rO2T
// SIG // ul3IIVPCglNjrxINUIcRGz1UUpaKLJw9khoImgUux5Ol
// SIG // SJHTMYIEcjCCBG4CAQEwgckwgbQxCzAJBgNVBAYTAlVT
// SIG // MRcwFQYDVQQKEw5WZXJpU2lnbiwgSW5jLjEfMB0GA1UE
// SIG // CxMWVmVyaVNpZ24gVHJ1c3QgTmV0d29yazE7MDkGA1UE
// SIG // CxMyVGVybXMgb2YgdXNlIGF0IGh0dHBzOi8vd3d3LnZl
// SIG // cmlzaWduLmNvbS9ycGEgKGMpMTAxLjAsBgNVBAMTJVZl
// SIG // cmlTaWduIENsYXNzIDMgQ29kZSBTaWduaW5nIDIwMTAg
// SIG // Q0ECECnBWz+qzVJqTme9PE5+P/IwCQYFKw4DAhoFAKBw
// SIG // MBAGCisGAQQBgjcCAQwxAjAAMBkGCSqGSIb3DQEJAzEM
// SIG // BgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsxDjAMBgor
// SIG // BgEEAYI3AgEVMCMGCSqGSIb3DQEJBDEWBBQsqBHrP0ok
// SIG // 2dnWJln1Ti7ld1t2mDANBgkqhkiG9w0BAQEFAASCAQBN
// SIG // M5PeztegV/d5vMmb/IuxrYWLo7oxDoU4lt6g42g/MWCj
// SIG // 9oV9aIewBxDlh4cVOrzQ8lfnA6Nizfqzj8QxSiW4ZRza
// SIG // LsV809E//N17+S8OXpDyiO6Sw9rIGkvk608X0XG6Zgx5
// SIG // B5wtrNOhL+F1gyQhFHCPqh1cMGxMhAX4iIubCzpFrgF0
// SIG // mM0afT7U7cDIhtgl2H5fSZDeGkZB3M9e7qxxuyNNlLri
// SIG // ZsE4xef3gFqhezQwVBy1PI7qCqYdQKU1z0en6oBds0wd
// SIG // OBU20DM5vXBMVzPFOi3SWKfVhSXTGb09lMLQHH2Hl6Zr
// SIG // CxHy6J9f+PcmSN8rMoqI0P4vvbVOHmwuoYICCzCCAgcG
// SIG // CSqGSIb3DQEJBjGCAfgwggH0AgEBMHIwXjELMAkGA1UE
// SIG // BhMCVVMxHTAbBgNVBAoTFFN5bWFudGVjIENvcnBvcmF0
// SIG // aW9uMTAwLgYDVQQDEydTeW1hbnRlYyBUaW1lIFN0YW1w
// SIG // aW5nIFNlcnZpY2VzIENBIC0gRzICEA7P9DjI/r81bgTY
// SIG // apgbGlAwCQYFKw4DAhoFAKBdMBgGCSqGSIb3DQEJAzEL
// SIG // BgkqhkiG9w0BBwEwHAYJKoZIhvcNAQkFMQ8XDTE0MDIw
// SIG // NzExMTQwOFowIwYJKoZIhvcNAQkEMRYEFFPVI08dzzt/
// SIG // BnSHWoHW/B2xzrBgMA0GCSqGSIb3DQEBAQUABIIBAD4E
// SIG // aYltliqzyj8SYeDI9wBSyC6HFRgwMChfcliJrZQFh2Cw
// SIG // g2rwsQssGWpe3dwRqE+PI9owsQ7fMuWKhwKtJ09mlL64
// SIG // cvnd/pJQ5QMCWZrb41sjISlx2/LgEXwGxCp63yknDN+V
// SIG // 83/EQQcMQRJHQA4k4yV0OJarKqnqwld3NPY/1w8Rkwcu
// SIG // cmv6dlXWKztibz7ui11VtdkJ4LrAcM9QZxc9bDDf0X2a
// SIG // wskZpDyob5oVK5S8BhjEJnSHJIFhBj8xPzKcIMPTai0U
// SIG // bv/WbyVu9a+rsgSG36GnJE2977OCwt9bCR9gvbCdtTAV
// SIG // Wri+1xEvhRK6hM/cnfja/qOa/JB+9Ls=
// SIG // End signature block
